/*
 * Analyser.java 1.0
 * 
 * Created on 23.03.2005.
 *
 * No copyright information available.
 */

import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.StringSelection;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.StringTokenizer;

/**
 * The snEADy Analyser is a very small tool to evaluate several round data from
 * snEADy.<br><br>
 * If the file "logs/result.log" has changed in the home directory from snEADy
 * all player names and points were read from this file. After that the average
 * ranks were calculated and copied to the clipboard.<br><br>
 * This tool may be useful to assess and improve your own snake. All you have
 * to do is to start the batch file (check that the location of the result.log
 * is right) and than start the snEADy game several times. You might do this
 * also with a batch file.<br><br>
 * Any questions? There's no need to be shy - mail to tthuem[at]web.de.
 * 
 * @version 1.0
 * @author Thomas Thm
 */
public class Analyser {
    
    /**
     * Count of milliseconds the modified time of the log file was checked.
     */
    private final static int UPDATE_CYCLE = 1000;
    
    /**
     * The specified log file.
     */
    private static File logFile;
    
    /**
     * Last time the log file was modified.
     */
    private static long lastModified; 

    /**
     * Starts the application with the location of the "logs/result.log" as a
     * parameter
     * 
     * @param  args  should contain only one parameter
     */
    public static void main(String[] args) {
        
        System.out.println();
        System.out.println("snEADy Analyser 1.0 (by Thomas Thuem)");
        System.out.println();
        
        if (args.length < 1) {
            throw new RuntimeException("Please specify the location of 'result.log'!");
        }
        logFile = new File(args[0]);
        if (!logFile.exists()) {
            throw new RuntimeException("The specified 'result.log' doesn't exist!");
        }
        
        lastModified = logFile.lastModified();
        
        while (true) {
            checkFileChanges();
            try {
                Thread.sleep(UPDATE_CYCLE);
            } catch (InterruptedException e) {
                e.printStackTrace();
            }
        }

    }
    
    /**
     * Checks if the log file has changed and reads the new round data.
     */
    private static void checkFileChanges() {
        if (logFile.lastModified() > lastModified) {
            readNextRound();
            printTournamentData();
            lastModified = logFile.lastModified();
        }
    }

    /**
     * Maximum count of players in the tournament.
     */
    private final static int MAX_PLAYER = 200;
    
    /**
     * Count of players already happened in this tournament.
     */
    private static int playerCount = 0;
    
    /**
     * Count of rounds in this tournament.
     */
    private static int roundCount = 0;
    
    /**
     * Names of all happend players.
     */
    private static String[] playerName = new String[MAX_PLAYER];
    
    /**
     * Average rank of all happend players.
     */
    private static double[] averageRank = new double[MAX_PLAYER];
    
    /**
     * Cumulated points of the players.
     */
    private static int[] points = new int[MAX_PLAYER];
    
    /**
     * Count of the times a player was registerd in this tournament.
     */
    private static int[] playTimes = new int[MAX_PLAYER];
    
    /**
     * Reads the round data from the specified log file.
     */
    private static void readNextRound() {
        
        roundCount++;
        
        try {

            BufferedReader input = new BufferedReader(new FileReader(logFile));
            
            int playerCount = 0;
            String[] playerName = new String[MAX_PLAYER];
            int[] points = new int[MAX_PLAYER];
            int[] rank = new int[MAX_PLAYER];
            
            //***************************************************************//
            // read player names and points                                  //
            //***************************************************************//
            
            String line;
            while (true) {
                
                while ((line = input.readLine()) != null && !line.startsWith("player"));
                if (line == null) break;
                
                //read the player name
                StringTokenizer tokenizer = new StringTokenizer(input.readLine(), ":");
                tokenizer.nextToken();
                playerName[playerCount] = tokenizer.nextToken().trim();
                
                //read the points
                input.readLine();
                tokenizer = new StringTokenizer(input.readLine(), ":");
                tokenizer.nextToken();
                points[playerCount] = Integer.parseInt(tokenizer.nextToken().trim());
                
                //increase player counter
                playerCount++;
                
            }
            
            //***************************************************************//
            // calculate ranks                                               //
            //***************************************************************//
            
            for (int i = 0; i < playerCount; i++) {
                rank[i] = 1;
            }
            for (int j = 1; j <= playerCount; j++) {
                int best = -1;
                for (int i = 0; i < playerCount; i++) {
                    if (rank[i] == j && (best < 0 || points[i] > points[best])) {
                        best = i;
                    }
                }
                if (best < 0) break;
                for (int i = 0; i < playerCount; i++) {
                    if (rank[i] == j && points[i] < points[best]) {
                        rank[i]++;
                    }
                }
            }
            
            //***************************************************************//
            // print round data                                              //
            //***************************************************************//
            
            //sort data
            int[] x = new int[playerCount];
            for (int i = 0; i < playerCount; i++) {
                x[i] = i;
            }
            for (int i = 0; i < playerCount - 1; i++) {
                int max = i;
                for (int j = i + 1; j < playerCount; j++) {
                    if (points[x[j]] > points[x[max]]) {
                        max = j;
                    }
                }
                int temp = x[i];
                x[i] = x[max];
                x[max] = temp;
            }
            for (int i = 0; i < playerCount - 1; i++) {
                int min = i;
                for (int j = i + 1; j < playerCount; j++) {
                    if (rank[x[j]] < rank[x[min]]) {
                        min = j;
                    }
                }
                int temp = x[i];
                x[i] = x[min];
                x[min] = temp;
            }
            
            //print tournament data
            StringBuffer output = new StringBuffer();
            output.append(".:. Round ");
            output.append(roundCount);
            output.append(" .:.\r\n");
            output.append("Rank\tPoints\tPlayer\r\n");
            for (int i = 0; i < playerCount; i++) {
                output.append(rank[x[i]]);
                output.append("\t");
                output.append(points[x[i]]);
                output.append("\t");
                output.append(playerName[x[i]]);
                output.append("\r\n");
            }
            System.out.print(output);
            System.out.println();
            
            //***************************************************************//
            // save round data                                               //
            //***************************************************************//
            
            for (int i = 0; i < playerCount; i++) {
                boolean saved = false;
                for (int j = 0; j < Analyser.playerCount; j++) {
                    if (playerName[i].equals(Analyser.playerName[j])) {
                        averageRank[j] *= playTimes[j];
                        averageRank[j] += rank[i]; 
                        playTimes[j]++;
                        averageRank[j] /= playTimes[j];
                        Analyser.points[j] += points[i];
                        saved = true;
                        break;
                    }
                }
                if (!saved) {
                    int j = Analyser.playerCount;
                    Analyser.playerName[j] = playerName[i];
                    averageRank[j] = rank[i]; 
                    playTimes[j] = 1;
                    Analyser.points[j] = points[i];
                    Analyser.playerCount++;
                }
            }
            
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        
    }

    /**
     * Prints the current tournament data and copies it to the clipboard.
     */
    private static void printTournamentData() {
        
        //sort data
        int[] x = new int[playerCount];
        for (int i = 0; i < playerCount; i++) {
            x[i] = i;
        }
        for (int i = 0; i < playerCount - 1; i++) {
            int max = i;
            for (int j = i + 1; j < playerCount; j++) {
                if (points[x[j]] > points[x[max]]) {
                    max = j;
                }
            }
            int temp = x[i];
            x[i] = x[max];
            x[max] = temp;
        }
        for (int i = 0; i < playerCount - 1; i++) {
            int min = i;
            for (int j = i + 1; j < playerCount; j++) {
                if (averageRank[x[j]] < averageRank[x[min]]) {
                    min = j;
                }
            }
            int temp = x[i];
            x[i] = x[min];
            x[min] = temp;
        }
        
        //print tournament data
        StringBuffer output = new StringBuffer();
        output.append(".:. Tournament .:.\r\n");
        output.append("Rank\tPoints\tRounds\tPlayer\r\n");
        for (int i = 0; i < playerCount; i++) {
            output.append((int) (averageRank[x[i]] * 10) / 10.0);
            output.append("\t");
            output.append(points[x[i]]);
            output.append("\t");
            output.append(playTimes[x[i]]);
            output.append("\t");
            output.append(playerName[x[i]]);
            output.append("\r\n");
        }
        System.out.print(output);
        System.out.println();
        
        //copy to clipboard
		setClipboardContents(output.toString());
        System.out.println("(Copied to clipboard.)");
        System.out.println();
        
    }

    /**
     * Places a string on the clipboard.
     */
    public static void setClipboardContents(String content) {
        StringSelection stringSelection = new StringSelection(content);
        Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
        clipboard.setContents(stringSelection, null);
    }

}
